/*
 * SD320_Command.cpp
 *
 *  Created on: Oct 29, 2020
 *      Author: Litvin
 */

#include "SD320_Command.h"

/// <summary>
/// Buffer storing generated packet
/// </summary>
uint8_t								SD320_Command::Buffer[253];
uint8_t								SD320_Command::PacketSize = 253;
uint16_t							SD320_Command::LastFormedCommandID;
float								SD320_Command::X_J1;
float								SD320_Command::Y_J2;
float								SD320_Command::Z_J3;
float								SD320_Command::ExtruderTemperature;
float								SD320_Command::TableTemperature;
float								SD320_Command::_3DPrinter_WheelRadiusMM;

uint16_t							SD320_Command::FirmWareVersion_FullRewriteVersion;
uint16_t							SD320_Command::FirmWareVersion_ChangeArchitectureVersion;
uint16_t							SD320_Command::FirmWareVersion_NewFunctionalityVersion;
uint16_t							SD320_Command::FirmWareVersion_BugFixingVersion;

ThreeFloatArgsFunctionPointer		SD320_Command::SetMaxSpeed_Handler;
ThreeFloatArgsFunctionPointer		SD320_Command::SetAcceleration_Handler;

ThreeFloatArgsFunctionPointer		SD320_Command::Move_Handler;
ThreeFloatArgsFunctionPointer		SD320_Command::MoveTo_Handler;
ThreeFloatArgsFunctionPointer		SD320_Command::JumpTo_Handler;
VoidArgsFunctionPointer				SD320_Command::CurrentPosition_Request_Handler;
ThreeFloatArgsFunctionPointer		SD320_Command::CurrentPosition_Response_Handler;

ThreeFloatArgsFunctionPointer		SD320_Command::jMove_Handler;
ThreeFloatArgsFunctionPointer		SD320_Command::jMoveTo_Handler;
VoidArgsFunctionPointer				SD320_Command::jCurrentPosition_Request_Handler;
ThreeFloatArgsFunctionPointer		SD320_Command::jCurrentPosition_Response_Handler;

OneBoolArgsFunctionPointer			SD320_Command::Tool_SetVacuumeCup_Handler;
OneBoolArgsFunctionPointer			SD320_Command::Tool_SetClaws_Handler;
OneUint8tArgsFunctionPointer		SD320_Command::Tool_SetLaserPWM_Handler;
OneFloatArgsFunctionPointer			SD320_Command::Tool_Rotate_Handler;
OneFloatArgsFunctionPointer			SD320_Command::Tool_RotateTo_Handler;
ThreeFloatArgsFunctionPointer		SD320_Command::Tool_SetTool_Handler;

VoidArgsFunctionPointer				SD320_Command::RobotCalibrate_Handler;


/*
 * Дополнительные команды для работы в режиме 3D принтера
 */
OneFloatArgsFunctionPointer			SD320_Command::_3DPrinter_SetFilamentLengthMMToRealease_Handler;
OneFloatArgsFunctionPointer			SD320_Command::_3DPrinter_SetLinearSpeedMmPerSec_Handler;

TwoFloatArgsFunctionPointer			SD320_Command::_3DPrinter_SetExtruderAndTableTemperatureCelsius_Handler;
VoidArgsFunctionPointer				SD320_Command::_3DPrinter_GetExtruderAndTableTemperatureCelsius_Request_Handler;
TwoFloatArgsFunctionPointer			SD320_Command::_3DPrinter_GetExtruderAndTableTemperatureCelsius_Response_Handler;
OneFloatArgsFunctionPointer			SD320_Command::_3DPrinter_SetAbsolutePositionFilamentStepper_Handler;
OneFloatArgsFunctionPointer			SD320_Command::_3DPrinter_SetWheelRadiusMM_Handler;
VoidArgsFunctionPointer				SD320_Command::_3DPrinter_GetWheelRadiusMM_Request_Handler;
OneFloatArgsFunctionPointer			SD320_Command::_3DPrinter_GetWheelRadiusMM_Response_Handler;
/*
 * Дополнительные команды для работы в режиме 3D принтера
 */

/*
 * Обработчики команд установки смещений для калибровки
 */
VoidArgsFunctionPointer		SD320_Command::SaveAxis1Offset_Handler;
VoidArgsFunctionPointer		SD320_Command::SaveAxis2Offset_Handler;
VoidArgsFunctionPointer		SD320_Command::SaveAxis3Offset_Handler;
VoidArgsFunctionPointer		SD320_Command::ResetAxis1Offset_Handler;
VoidArgsFunctionPointer		SD320_Command::ResetAxis2Offset_Handler;
VoidArgsFunctionPointer		SD320_Command::ResetAxis3Offset_Handler;
/*
 * Обработчики команд установки смещений для калибровки
 */


VoidArgsFunctionPointer				SD320_Command::GetFirmwareVersion_Request_Handler;
FourUint16ArgsFunctionPointer		SD320_Command::GetFirmwareVersion_Response_Handler;
OneBoolArgsFunctionPointer			SD320_Command::CommandDone_Handler;
bool								SD320_Command::CommandDoneFlag;


/// <summary>
/// Class initialization for master SPI
/// </summary>
void SD320_Command::InitForMaster()
{
	CommandDone_Handler =
	[](bool isDone)->uint8_t
	{
		#if defined(DEBUG) && defined(ARDUINO)
			DEBUGPRINTLN("Подтверждение получения команды получено");
		#endif
		CommandDoneFlag = isDone;
		return 1;
	};

	CurrentPosition_Response_Handler = [](float x, float y, float z)->uint8_t
	{
		X_J1 = x;
		Y_J2 = y;
		Z_J3 = z;

		CommandDoneFlag = true;

		#if defined(DEBUG) && defined(ARDUINO)
			DEBUGPRINTLN("Текущие координаты синхронизированы. Новые значения:");
			DEBUGPRINTLN_WITHTEXT("x: ", X_J1);
			DEBUGPRINTLN_WITHTEXT("y: ", Y_J2);
			DEBUGPRINTLN_WITHTEXT("z: ", Z_J3);
		#endif
		return 1;
	};

	jCurrentPosition_Response_Handler = [](float j1, float j2, float j3)->uint8_t
	{
		X_J1 = j1;
		Y_J2 = j2;
		Z_J3 = j3;

		CommandDoneFlag = true;

		#if defined(DEBUG) && defined(ARDUINO)
			DEBUGPRINTLN("Текущие углы синхронизированы. Новые значения:");
			DEBUGPRINTLN_WITHTEXT("j1: ", X_J1);
			DEBUGPRINTLN_WITHTEXT("j2: ", Y_J2);
			DEBUGPRINTLN_WITHTEXT("j3: ", Z_J3);
		#endif
		return 1;
	};

	GetFirmwareVersion_Response_Handler = [](uint16_t FullRewriteVersion, uint16_t ChangeArchitectureVersion, uint16_t NewFunctionalityVersion, uint16_t BugFixingVersion)->uint8_t
	{
		FirmWareVersion_FullRewriteVersion = FullRewriteVersion;
		FirmWareVersion_ChangeArchitectureVersion = ChangeArchitectureVersion;
		FirmWareVersion_NewFunctionalityVersion = NewFunctionalityVersion;
		FirmWareVersion_BugFixingVersion = BugFixingVersion;

		CommandDoneFlag = true;

		return 1;
	};

	_3DPrinter_GetExtruderAndTableTemperatureCelsius_Response_Handler = [](float extruderTemperature, float tableTemperature)->uint8_t
	{
		ExtruderTemperature = extruderTemperature;
		TableTemperature = tableTemperature;

		CommandDoneFlag = true;

		return 1;
	};

	_3DPrinter_GetWheelRadiusMM_Response_Handler = [](float value)->uint8_t
	{
		_3DPrinter_WheelRadiusMM = value;

		CommandDoneFlag = true;
		return 1;
	};
}

/// <summary>
/// Handle received packet
/// </summary>
/// <param name="buf">Received byte packet</param>
/// <returns>Returns 1 if handling correctly completed, otherwise - 0</returns>
uint8_t SD320_Command::HandlePacket(uint8_t* buf)
{
	uint16_t command = (((uint16_t)buf[0]) << 8) | buf[1];
	uint8_t result = 0;
	uint8_t packetLength = buf[2];

	// Checking CRC of received packet
	if(!Simple_CRC::CheckOneByteXorCRC(buf, packetLength - 1, buf[packetLength - 1]))
		return 0;

	float x, y, z;
	uint16_t a, b, c, d;
	bool flag = false;
	uint8_t p;

	switch (command)
	{
		case SD320_Command_SetMaxSpeed:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if (SetMaxSpeed_Handler != NULL)
				result = SetMaxSpeed_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_SetAcceleration:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if (SetAcceleration_Handler != NULL)
				result = SetAcceleration_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_Move:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if (Move_Handler != NULL)
				result = Move_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_MoveTo:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if(MoveTo_Handler != NULL)
				result = MoveTo_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_JumpTo:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if (JumpTo_Handler != NULL)
				result = JumpTo_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_CurrentPosition_Request:
			if (CurrentPosition_Request_Handler != NULL)
				result = CurrentPosition_Request_Handler();
			else
				result = 0;
			break;

		case SD320_Command_CurrentPosition_Response:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if (CurrentPosition_Response_Handler != NULL)
				result = CurrentPosition_Response_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_jMove:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if (jMove_Handler != NULL)
				result = jMove_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_jMoveTo:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if(jMoveTo_Handler != NULL)
				result = jMoveTo_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_jCurrentPosition_Request:
			if (jCurrentPosition_Request_Handler != NULL)
				result = jCurrentPosition_Request_Handler();
			else
				result = 0;
			break;

		case SD320_Command_jCurrentPosition_Response:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if (jCurrentPosition_Response_Handler != NULL)
				result = jCurrentPosition_Response_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_Tool_SetVacuumeCup:
			flag = *((bool*)&buf[3]);
			if(Tool_SetVacuumeCup_Handler != NULL)
				result = Tool_SetVacuumeCup_Handler(flag);
			else
				result = 0;
			break;

		case SD320_Command_Tool_SetClaws:
			flag = *((bool*)&buf[3]);
			if (Tool_SetClaws_Handler != NULL)
				result = Tool_SetClaws_Handler(flag);
			else
				result = 0;
			break;

		case SD320_Command_Tool_SetLaserPWM:
			p = *((uint8_t*)&buf[3]);
			if (Tool_SetLaserPWM_Handler != NULL)
				result = Tool_SetLaserPWM_Handler(p);
			else
				result = 0;
			break;

		case SD320_Command_Tool_Rotate:
			x = *((float*)&buf[3]);
			if(Tool_Rotate_Handler != NULL)
				result = Tool_Rotate_Handler(x);
			else
				result = 0;
			break;

		case SD320_Command_Tool_RotateTo:
			x = *((float*)&buf[3]);
			if(Tool_RotateTo_Handler != NULL)
				result = Tool_RotateTo_Handler(x);
			else
				result = 0;
			break;

		case SD320_Command_Tool_SetTool:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			z = *((float*)&buf[11]);
			if (Tool_SetTool_Handler != NULL)
				result = Tool_SetTool_Handler(x, y, z);
			else
				result = 0;
			break;

		case SD320_Command_RobotCalibrate:
			if(RobotCalibrate_Handler != NULL)
				result = RobotCalibrate_Handler();
			else
				result = 0;
			break;

		case SD320_Command_GetFirmwareVersion_Request:
			if (GetFirmwareVersion_Request_Handler != NULL)
				result = GetFirmwareVersion_Request_Handler();
			else
				result = 0;
			break;

		case SD320_Command_GetFirmwareVersion_Response:
			a = *((uint16_t*)&buf[3]);
			b = *((uint16_t*)&buf[5]);
			c = *((uint16_t*)&buf[7]);
			d = *((uint16_t*)&buf[9]);
			if (GetFirmwareVersion_Response_Handler != NULL)
				result = GetFirmwareVersion_Response_Handler(a,b,c,d);
			else
				result = 0;
			break;

		case SD320_Command_CommandDone:
			flag = *((bool*)&buf[3]);
			if(CommandDone_Handler != NULL)
				result = CommandDone_Handler(flag);
			else
				result = 0;
			break;

		/*
		 * Дополнительные команды для работы в режиме 3D принтера
		 */
		case SD320_Command_3DPrinter_SetFilamentLengthMMToRealease:
			x = *((float*)&buf[3]);
			if(_3DPrinter_SetFilamentLengthMMToRealease_Handler != NULL)
				result = _3DPrinter_SetFilamentLengthMMToRealease_Handler(x);
			else
				result = 0;
			break;

		case SD320_Command_3DPrinter_SetLinearSpeedMmPerSec:
			x = *((float*)&buf[3]);
			if(_3DPrinter_SetLinearSpeedMmPerSec_Handler != NULL)
				result = _3DPrinter_SetLinearSpeedMmPerSec_Handler(x);
			else
				result = 0;
			break;

		case SD320_Command_3DPrinter_SetExtruderAndTableTemperatureCelsius:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			if(_3DPrinter_SetExtruderAndTableTemperatureCelsius_Handler != NULL)
				result = _3DPrinter_SetExtruderAndTableTemperatureCelsius_Handler(x, y);
			else
				result = 0;
			break;

		case SD320_Command_3DPrinter_GetExtruderAndTableTemperatureCelsius_Request:
			if(_3DPrinter_GetExtruderAndTableTemperatureCelsius_Request_Handler != NULL)
				result = _3DPrinter_GetExtruderAndTableTemperatureCelsius_Request_Handler();
			else
				result = 0;
			break;

		case SD320_Command_3DPrinter_GetExtruderAndTableTemperatureCelsius_Response:
			x = *((float*)&buf[3]);
			y = *((float*)&buf[7]);
			if(_3DPrinter_GetExtruderAndTableTemperatureCelsius_Response_Handler != NULL)
				result = _3DPrinter_GetExtruderAndTableTemperatureCelsius_Response_Handler(x, y);
			else
				result = 0;
			break;

		case SD320_Command_3DPrinter_SetAbsolutePositionFilamentStepper:
			x = *((float*)&buf[3]);
			if(_3DPrinter_SetAbsolutePositionFilamentStepper_Handler != NULL)
				result = _3DPrinter_SetAbsolutePositionFilamentStepper_Handler(x);
			else
				result = 0;
			break;

		case SD320_Command_3DPrinter_SetWheelRadiusMM:
			x = *((float*)&buf[3]);
			if(_3DPrinter_SetWheelRadiusMM_Handler != NULL)
				result = _3DPrinter_SetWheelRadiusMM_Handler(x);
			else
				result = 0;
			break;

		case SD320_Command_3DPrinter_GetWheelRadiusMM_Request:
			if(_3DPrinter_GetWheelRadiusMM_Request_Handler != NULL)
				result = _3DPrinter_GetWheelRadiusMM_Request_Handler();
			else
				result = 0;
			break;

		case SD320_Command_3DPrinter_GetWheelRadiusMM_Response:
			x = *((float*)&buf[3]);
			if(_3DPrinter_GetWheelRadiusMM_Response_Handler != NULL)
				result = _3DPrinter_GetWheelRadiusMM_Response_Handler(x);
			else
				result = 0;
			break;

		case SD320_Command_SaveAxis1Offset:
			if(SaveAxis1Offset_Handler != NULL)
				result = SaveAxis1Offset_Handler();
			else
				result = 0;
			break;

		case SD320_Command_SaveAxis2Offset:
			if(SaveAxis2Offset_Handler != NULL)
				result = SaveAxis2Offset_Handler();
			else
				result = 0;
			break;

		case SD320_Command_SaveAxis3Offset:
			if(SaveAxis3Offset_Handler != NULL)
				result = SaveAxis3Offset_Handler();
			else
				result = 0;
			break;

		case SD320_Command_ResetAxis1Offset:
			if(ResetAxis1Offset_Handler != NULL)
				result = ResetAxis1Offset_Handler();
			else
				result = 0;
			break;

		case SD320_Command_ResetAxis2Offset:
			if(ResetAxis2Offset_Handler != NULL)
				result = ResetAxis2Offset_Handler();
			else
				result = 0;
			break;


		case SD320_Command_ResetAxis3Offset:
			if(ResetAxis3Offset_Handler != NULL)
				result = ResetAxis3Offset_Handler();
			else
				result = 0;
			break;

		/*
		 * Дополнительные команды для работы в режиме 3D принтера
		 */
	}
	return result;
}


uint8_t* SD320_Command::SetMaxSpeed(float axis1Speed, float axis2Speed, float axis3Speed)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_SetMaxSpeed);

	ClearBuffer();
	SetCommandID(SD320_Command_SetMaxSpeed);

	AddValue((uint8_t*)&axis1Speed, 4);
	AddValue((uint8_t*)&axis2Speed, 4);
	AddValue((uint8_t*)&axis3Speed, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}


uint8_t* SD320_Command::SetAcceleration(float axis1Acceleration, float axis2Acceleration, float axis3Acceleration)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_SetAcceleration);

	ClearBuffer();
	SetCommandID(SD320_Command_SetAcceleration);

	AddValue((uint8_t*)&axis1Acceleration, 4);
	AddValue((uint8_t*)&axis2Acceleration, 4);
	AddValue((uint8_t*)&axis3Acceleration, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}


/// <summary>
/// Form the packet for relative move by axis
/// </summary>
/// <param name="x">Relative value of x axis</param>
/// <param name="y">Relative value of y axis</param>
/// <param name="z">Relative value of z axis</param>
/// <returns>Return packet size</returns>
uint8_t* SD320_Command::Move(float relative_X, float relative_Y, float relative_Z)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_Move);
	
	ClearBuffer();
	SetCommandID(SD320_Command_Move);

	AddValue((uint8_t*)&relative_X, 4);
	AddValue((uint8_t*)&relative_Y, 4);
	AddValue((uint8_t*)&relative_Z, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}
/// <summary>
/// Form the packet for relative move by axis
/// </summary>
/// <param name="x">Relative value of x axis</param>
/// <param name="y">Relative value of y axis</param>
/// <param name="z">Relative value of z axis</param>
/// <returns>Return packet size</returns>
uint8_t* SD320_Command::MoveTo(float absoluteX, float absoluteY, float absoluteZ)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_MoveTo);

	ClearBuffer();
	SetCommandID(SD320_Command_MoveTo);

	AddValue((uint8_t*)&absoluteX, 4);
	AddValue((uint8_t*)&absoluteY, 4);
	AddValue((uint8_t*)&absoluteZ, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::JumpTo(float absoluteX, float absoluteY, float deltaZ)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_MoveTo);

	ClearBuffer();
	SetCommandID(SD320_Command_JumpTo);

	AddValue((uint8_t*)&absoluteX, 4);
	AddValue((uint8_t*)&absoluteY, 4);
	AddValue((uint8_t*)&deltaZ, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}
/// <summary>
/// Формирует запрос на получение текущего положения
/// </summary>
/// <returns>Возвращает указатель на массив со сформированным пакетом</returns>
uint8_t* SD320_Command::CurrentPosition_Request(void)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_CurrentPosition_Request);

	ClearBuffer();
	SetCommandID(SD320_Command_CurrentPosition_Request);



	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::CurrentPosition_Response(float absoluteX, float absoluteY, float absoluteZ)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_CurrentPosition_Response);

	ClearBuffer();
	SetCommandID(SD320_Command_CurrentPosition_Response);

	AddValue((uint8_t*)&absoluteX, 4);
	AddValue((uint8_t*)&absoluteY, 4);
	AddValue((uint8_t*)&absoluteZ, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}

#define SD320_COMMAND_LIMIT_J1_MIN (0.0f)
#define SD320_COMMAND_LIMIT_J1_MAX (180.0f)
#define SD320_COMMAND_LIMIT_J2_MIN (-25.0f)
#define SD320_COMMAND_LIMIT_J2_MAX (100.0f)
#define SD320_COMMAND_LIMIT_J3_MIN (-120.0f)
#define SD320_COMMAND_LIMIT_J3_MAX (20.0f)

bool SD320_Command::isValidRelativeJoint(float J1, float J2, float J3)
{
	return	fabs(J1) <= (SD320_COMMAND_LIMIT_J1_MAX - SD320_COMMAND_LIMIT_J1_MIN) &&
			fabs(J2) <= (SD320_COMMAND_LIMIT_J2_MAX - SD320_COMMAND_LIMIT_J2_MIN) &&
			fabs(J3) <= (SD320_COMMAND_LIMIT_J3_MAX - SD320_COMMAND_LIMIT_J3_MIN);
};

bool SD320_Command::isValidAbsoluteJoint(float J1, float J2, float J3)
{
	#ifndef DEGTORAD
		#define DEGTORAD (M_PI / 180.0f)
	#endif
	#define L1 (135.0f)
	#define L2 (147.0f)

	return  (J1 >= SD320_COMMAND_LIMIT_J1_MIN && J1 <= SD320_COMMAND_LIMIT_J1_MAX) && // Ограничение оси 1 из документации
			(J2 >= SD320_COMMAND_LIMIT_J2_MIN && J2 <= SD320_COMMAND_LIMIT_J2_MAX) && // Ограничение оси 2 из документации
			(J3 >= SD320_COMMAND_LIMIT_J3_MIN && J3 <= SD320_COMMAND_LIMIT_J3_MAX) && // Ограничение оси 3 из документации
			(J2 + J3 <= 64) && // Ограничение на раскладывание
			(J2 + J3 >= -73) && // Ограничение на сложение
			(((L1 * sin(J2 * DEGTORAD) + L2 * cos(J3 * DEGTORAD)) >= 95) || ((L1 * cos(J2 * DEGTORAD) + L2 * sin(J3 * DEGTORAD)) > -86)); // Либо выше либо дальше основания
};

uint8_t* SD320_Command::jMove(float relative_j1, float relative_j2, float relative_j3)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_jMove);

	if(!isValidRelativeJoint(relative_j1, relative_j2, relative_j3))
		return NULL;

	ClearBuffer();
	SetCommandID(SD320_Command_jMove);

	AddValue((uint8_t*)&relative_j1, 4);
	AddValue((uint8_t*)&relative_j2, 4);
	AddValue((uint8_t*)&relative_j3, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}
/// <summary>
/// Form the packet for relative move by joints
/// </summary>
/// <param name="x">Relative value of j1</param>
/// <param name="y">Relative value of j2</param>
/// <param name="z">Relative value of j3</param>
/// <returns>Return packet size</returns>
uint8_t* SD320_Command::jMoveTo(float absolute_j1, float absolute_j2, float absolute_j3)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_jMoveTo);

	if(!isValidAbsoluteJoint(absolute_j1, absolute_j2, absolute_j3))
		return NULL;

	ClearBuffer();
	SetCommandID(SD320_Command_jMoveTo);

	AddValue((uint8_t*)&absolute_j1, 4);
	AddValue((uint8_t*)&absolute_j2, 4);
	AddValue((uint8_t*)&absolute_j3, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}
/// <summary>
/// Формирует запрос на получение текущего положения в углах
/// </summary>
/// <returns>Возвращает указатель на массив со сформированным пакетом</returns>
uint8_t* SD320_Command::jCurrentPosition_Request(void)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_jCurrentPosition_Request);

	ClearBuffer();
	SetCommandID(SD320_Command_jCurrentPosition_Request);



	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::jCurrentPosition_Response(float absoluteJ1, float absoluteJ2, float absoluteJ3)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_CurrentPosition_Response);

	ClearBuffer();
	SetCommandID(SD320_Command_CurrentPosition_Response);

	AddValue((uint8_t*)&absoluteJ1, 4);
	AddValue((uint8_t*)&absoluteJ2, 4);
	AddValue((uint8_t*)&absoluteJ3, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}



/// <summary>
/// Form the packet for tool rotating
/// </summary>
/// <param name="angle">Angle for tool</param>
/// <returns>Return packet size</returns>
uint8_t* SD320_Command::Tool_RotateTo(float absolute)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_Tool_RotateTo);

	ClearBuffer();
	SetCommandID(SD320_Command_Tool_RotateTo);

	AddValue((uint8_t*)&absolute, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::Tool_Rotate(float relative)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_Tool_Rotate);

	ClearBuffer();
	SetCommandID(SD320_Command_Tool_Rotate);

	AddValue((uint8_t*)&relative, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::Tool_SetVacuumeCup(bool isEnable)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_Tool_SetVacuumeCup);

	ClearBuffer();
	SetCommandID(SD320_Command_Tool_SetVacuumeCup);

	AddValue((uint8_t*)&isEnable, 1);

	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::Tool_SetClaws(bool isEnable)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_Tool_SetClaws);

	ClearBuffer();
	SetCommandID(SD320_Command_Tool_SetClaws);

	AddValue((uint8_t*)&isEnable, 1);

	SetLength();
	CalculateCRC();
	return Buffer;
}

uint8_t* SD320_Command::Tool_SetLaserPWM(uint8_t pwm)
{
	ClearBuffer();
	SetCommandID(SD320_Command_Tool_SetLaserPWM);

	AddValue((uint8_t*)&pwm, 1);

	SetLength();
	CalculateCRC();
	return Buffer;
}

uint8_t* SD320_Command::Tool_SetTool(float TX0, float TY0, float TZ0)
{
	ClearBuffer();
	SetCommandID(SD320_Command_Tool_SetTool);

	AddValue((uint8_t*)&TX0, 4);
	AddValue((uint8_t*)&TY0, 4);
	AddValue((uint8_t*)&TZ0, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}

/// <summary>
/// Form the packet for robot calibration
/// </summary>
/// <returns>Return packet size</returns>
uint8_t* SD320_Command::RobotCalibrate(void)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_RobotCalibrate);

	ClearBuffer();
	SetCommandID(SD320_Command_RobotCalibrate);

	SetLength();
	CalculateCRC();
	return Buffer;
}

uint8_t* SD320_Command::GetFirmwareVersion_Request(void)
{
	ClearBuffer();
	SetCommandID(SD320_Command_GetFirmwareVersion_Request);

	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::GetFirmwareVersion_Response(uint16_t FullRewriteVersion, uint16_t ChangeArchitectureVersion, uint16_t NewFunctionalityVersion, uint16_t BugFixingVersion)
{
	ClearBuffer();
	SetCommandID(SD320_Command_GetFirmwareVersion_Response);

	AddValue((uint8_t*)&FullRewriteVersion, 2);
	AddValue((uint8_t*)&ChangeArchitectureVersion, 2);
	AddValue((uint8_t*)&NewFunctionalityVersion, 2);
	AddValue((uint8_t*)&BugFixingVersion, 2);

	SetLength();
	CalculateCRC();
	return Buffer;
}
/// <summary>
/// Form the packet for command done response (for slave)
/// </summary>
/// <returns>Return packet size</returns>
uint8_t* SD320_Command::CommandDone(bool isDone)
{
	//INC_SD320_COMMAND_CheckLastFormedCommandID(SD320_Command_CommandDone);

	ClearBuffer();
	SetCommandID(SD320_Command_CommandDone);

	AddValue((uint8_t*)&isDone, 1);

	SetLength();
	CalculateCRC();
	return Buffer;
}

/*
 * Дополнительные команды для работы в режиме 3D принтера
 */
uint8_t* SD320_Command::_3DPrinter_SetFilamentLengthMMToRealease(float LengthMm)
{
	ClearBuffer();
	SetCommandID(SD320_Command_3DPrinter_SetFilamentLengthMMToRealease);

	AddValue((uint8_t*)&LengthMm, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}

uint8_t* SD320_Command::_3DPrinter_SetLinearSpeedMmPerSec(float SpeedMmPerSec)
{
	ClearBuffer();
	SetCommandID(SD320_Command_3DPrinter_SetLinearSpeedMmPerSec);

	AddValue((uint8_t*)&SpeedMmPerSec, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}


uint8_t* SD320_Command::_3DPrinter_SetExtruderAndTableTemperatureCelsius(float extruderTemperature, float tableTemperature)
{
	ClearBuffer();
	SetCommandID(SD320_Command_3DPrinter_SetExtruderAndTableTemperatureCelsius);

	AddValue((uint8_t*)&extruderTemperature, 4);
	AddValue((uint8_t*)&tableTemperature, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}

uint8_t* SD320_Command::_3DPrinter_GetExtruderAndTableTemperatureCelsius_Request(void)
{
	ClearBuffer();
	SetCommandID(SD320_Command_3DPrinter_GetExtruderAndTableTemperatureCelsius_Request);



	SetLength();
	CalculateCRC();
	return Buffer;
}

uint8_t* SD320_Command::_3DPrinter_GetExtruderAndTableTemperatureCelsius_Response(float extruderTemperature, float tableTemperature)
{
	ClearBuffer();
	SetCommandID(SD320_Command_3DPrinter_GetExtruderAndTableTemperatureCelsius_Response);

	AddValue((uint8_t*)&extruderTemperature, 4);
	AddValue((uint8_t*)&tableTemperature, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}

uint8_t* SD320_Command::_3DPrinter_SetAbsolutePositionFilamentStepper(float absoluteMm)
{
	ClearBuffer();
	SetCommandID(SD320_Command_3DPrinter_SetAbsolutePositionFilamentStepper);

	AddValue((uint8_t*)&absoluteMm, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}


uint8_t* SD320_Command::_3DPrinter_SetWheelRadiusMM(float value)
{
	ClearBuffer();
	SetCommandID(SD320_Command_3DPrinter_SetWheelRadiusMM);

	AddValue((uint8_t*)&value, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}

uint8_t* SD320_Command::_3DPrinter_GetWheelRadiusMM_Request()
{
	ClearBuffer();
	SetCommandID(SD320_Command_3DPrinter_GetWheelRadiusMM_Request);

	SetLength();
	CalculateCRC();
	return Buffer;
}

uint8_t* SD320_Command::_3DPrinter_GetWheelRadiusMM_Response(float value)
{
	ClearBuffer();
	SetCommandID(SD320_Command_3DPrinter_GetWheelRadiusMM_Response);

	AddValue((uint8_t*)&value, 4);

	SetLength();
	CalculateCRC();
	return Buffer;
}
/*
 * Дополнительные команды для работы в режиме 3D принтера
 */


/*
 * Функции формирования команд установки смещений для калибровки
 */
uint8_t* SD320_Command::SaveAxis1Offset(void)
{
	ClearBuffer();
	SetCommandID(SD320_Command_SaveAxis1Offset);
	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::SaveAxis2Offset(void)
{
	ClearBuffer();
	SetCommandID(SD320_Command_SaveAxis2Offset);
	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::SaveAxis3Offset(void)
{
	ClearBuffer();
	SetCommandID(SD320_Command_SaveAxis3Offset);
	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::ResetAxis1Offset(void)
{
	ClearBuffer();
	SetCommandID(SD320_Command_ResetAxis1Offset);
	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::ResetAxis2Offset(void)
{
	ClearBuffer();
	SetCommandID(SD320_Command_ResetAxis2Offset);
	SetLength();
	CalculateCRC();
	return Buffer;
}
uint8_t* SD320_Command::ResetAxis3Offset(void)
{
	ClearBuffer();
	SetCommandID(SD320_Command_ResetAxis3Offset);
	SetLength();
	CalculateCRC();
	return Buffer;
}
/*
 * Функции формирования команд установки смещений для калибровки
 */










/// <summary>
/// Clear main buffer.
/// </summary>
void SD320_Command::ClearBuffer()
{
	for (uint8_t i = 0; i < PacketSize; i++)
		Buffer[i] = 0;
	PacketSize = 0;
}

/// <summary>
/// Sets in buffer command ID
/// </summary>
/// <param name="command">command ID</param>
void SD320_Command::SetCommandID(uint16_t command)
{
	Buffer[0] = (uint8_t)(command >> 8);
	Buffer[1] = (uint8_t)command;
	PacketSize = 3;
	LastFormedCommandID = command;
	CommandDoneFlag = false;
}

/// <summary>
/// Added to buffer value from the value pointer of the current size
/// </summary>
/// <param name="value">Pointer to value</param>
/// <param name="valueByteSize">Value byte count</param>
void SD320_Command::AddValue(uint8_t* value, uint8_t valueByteSize)
{
	for (uint8_t i = 0; i < valueByteSize; i++)
		Buffer[PacketSize + i] = value[i];
	PacketSize += valueByteSize;
}

void SD320_Command::SetLength()
{
	Buffer[2] = ++PacketSize; // +1 for crc
}

/// <summary>
/// Calculate and added to buffer crc checksum
/// </summary>
void SD320_Command::CalculateCRC()
{
	Buffer[PacketSize-1] = Simple_CRC::CalculateOneByteXorCRC(Buffer, PacketSize-1);
	#if defined(DEBUG) && defined(ARDUINO)
		Serial.print("Packet CRC: "); Serial.println(Buffer[PacketSize - 1], HEX);
	#endif
}
